<?php

namespace App\Http\Controllers;

use App\Enums\BookingStatus;
use App\Enums\ShippingMethod;
use App\Http\Requests\Booking\TransitRequest;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\InTransit;
use App\Models\InTransitSummary;
use App\Models\User;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class TransitController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        $branchId = auth()->user()->default_branch_id;
        $customer = Customer::active()
            ->where('branch_id', $branchId)
            ->select('id', 'customer_name', 'phone', 'currency', 'ref_code')
            ->get();
        $warehouse = Warehouse::get()->pluck('name', 'id');
        $shipping = collect(ShippingMethod::cases())->map(function ($status) {
            return [
                'value' => $status->value,
                'label' => $status->label(),
            ];
        });
        $driver = User::get()->pluck('display_name', 'id');

        return view('pages.booking.transit', compact('customer', 'warehouse', 'shipping', 'driver'));
    }

    /**
     * Display a listing of the resource.
     */
    public function transitAjax(Request $request)
    {
        $branchId = $request->user()->default_branch_id;

        if ($request->ajax()) {
            $data = Delivery::where('branch_id', $branchId)
                ->whereIn('status', [
                    BookingStatus::ARRIVEDWAREHOUSE->value,
                    BookingStatus::ARRIVEDWAREHOUSEPP->value,
                ])
                ->when($request->from_warehouse, function ($query, $from_warehouse) {
                    return $query->where('from_warehouse_id', $from_warehouse);
                })
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->shipping_method, function ($query, $shipping_method) {
                    return $query->where('shipping_method', $shipping_method);
                })
                ->when($request->entry_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->entry_at);

                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('created_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        return $query->whereBetween('created_at', [
                            $dates[0].' 00:00:00',
                            $dates[0].' 23:59:59',
                        ]);
                    }

                    return $query;
                })

                ->when($request->completed_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->completed_at);

                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('completed_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        $date = $dates[0];

                        return $query->whereBetween('completed_at', [
                            $date.' 00:00:00',
                            $date.' 23:59:59',
                        ]);
                    }

                    return $query;
                })

                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('ref_code', function ($row) {
                    return $row->customer->ref_code ?? 'N/A';
                })
                ->addColumn('photo', function ($row) {
                    return $row->photo ? asset('storage/'.$row->photo) : null;
                })

                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('from_warehouse_name', function ($row) {
                    return $row->fromWarehouse->name ?? 'N/A';
                })
                ->addColumn('to_warehouse_name', function ($row) {
                    return $row->toWarehouse->name ?? 'N/A';
                })

                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })
                ->addColumn('created_by', function ($row) {
                    return $row->createdBy->display_name ?? 'N/A';
                })
                ->editColumn('created_at', function ($row) {
                    if (! $row->created_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->created_at)->format('d-F-Y');
                })
                ->addColumn('status_label', function ($row) {
                    $status = BookingStatus::tryFrom($row->status);
                    $label = $status?->label() ?? 'Unknown';
                    $colorClass = match ($status) {
                        BookingStatus::BOOKING => 'text-warning',
                        BookingStatus::ARRIVEDWAREHOUSE => 'text-info',
                        BookingStatus::IN_TRANSIT => 'text-success',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                    HTML;
                })

                ->addColumn('action', function ($row) {
                    return view('pages.booking.action-dropdown', [
                        'row' => $row,
                        'canEdit' => $row->is_paid !== true,
                    ])->render();
                })

                ->addColumn('status_cancel', function ($row) {
                    $hasReason = $row->cancel_reason_id !== null;
                    $reason = $hasReason ? 'Yes' : 'No';
                    $colorClass = $hasReason ? 'text-error' : 'text-success';
                    $tooltipText = $hasReason ? $row->cancelReason->reason : 'No cancel reason';

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}" x-tooltip.delay.500="'{$tooltipText}'">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$reason}</span>
                        </div>
                    HTML;
                })

                ->rawColumns(['action', 'status_label', 'status_cancel'])
                ->make(true);

        }
    }

    public function store(TransitRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $deliveryIds = $data['delivery_ids'];
            $deliveryById = $data['truck_by'];
            $fromWarehouse = $data['from_warehouse'];
            $toWarehouse = $data['to_warehouse'];
            $shipping = $data['shipping_method'];

            $deliveries = Delivery::whereIn('id', $deliveryIds)
                ->whereIn('status', [
                    BookingStatus::ARRIVEDWAREHOUSE->value,
                    BookingStatus::ARRIVEDWAREHOUSEPP->value,
                ])
                ->get();

            $summary = InTransitSummary::create([
                'invoice_number' => 'DN'.now()->format('YmdHis'),
                'truck_id' => $deliveryById,
                'from_warehouse_id' => $fromWarehouse,
                'to_warehouse_id' => $toWarehouse,
                'shipping_method' => $shipping,
                'service_fee' => $deliveries->sum('amount_usd'),
                'status' => BookingStatus::IN_TRANSIT->value,
                'date' => now(),
                'created_by' => auth()->id(),
            ]);

            foreach ($deliveries as $delivery) {

                InTransit::create([
                    'in_transit_summary_id' => $summary->id,
                    'delivery_id' => $delivery->id,
                    'date' => now(),
                    'created_by' => auth()->id(),
                ]);

                $delivery->update([
                    'truck_id' => $deliveryById,
                    'status' => BookingStatus::IN_TRANSIT->value,
                    'updated_by' => auth()->id(),
                ]);

                DeliveryStatusHistory::create([
                    'delivery_id' => $delivery->id,
                    'status' => BookingStatus::IN_TRANSIT->value,
                    'changed_by' => auth()->id(),
                    'changed_at' => now(),
                    'note' => 'In Transit',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);
            }

            DB::commit();

            return back()->with('success', 'In Transit successfully.');

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Failed to in transit: '.$e->getMessage());

            return redirect()->back()->withInput()->with('error', 'Failed to in transit. Please try again.');
        }
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
