<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Enums\Status;
use App\Models\Customer;
use App\Models\Delivery;

class DashboardController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:dashboard.view')->only(['index']);
    }

    public function index()
    {

        $branchId = auth()->user()->default_branch_id;
        $monthlyFees = Delivery::selectRaw('
            MONTH(completed_at) as month,
            SUM(delivery_price) as total_fee
        ')
            ->where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_paid', true)
            ->whereNotNull('completed_at')
            ->groupByRaw('MONTH(completed_at)')
            ->orderByRaw('MONTH(completed_at)')
            ->get();

        $profit = array_fill(0, 12, 0);

        foreach ($monthlyFees as $item) {
            $profit[$item->month - 1] = (float) $item->total_fee;
        }

        $currentMonthTotal = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->whereNotNull('completed_at')
            ->whereMonth('completed_at', now()->month)
            ->whereYear('completed_at', now()->year)
            ->sum('delivery_price');

        $lastMonth = now()->subMonth();
        $lastMonthTotal = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->whereNotNull('completed_at')
            ->whereMonth('completed_at', $lastMonth->month)
            ->whereYear('completed_at', $lastMonth->year)
            ->sum('delivery_price');

        $percentageChange = null;
        $trend = null;

        if ($lastMonthTotal > 0) {
            $change = $currentMonthTotal - $lastMonthTotal;
            $percentageChange = round(($change / $lastMonthTotal) * 100, 1);
            $trend = $percentageChange > 0 ? 'up' : ($percentageChange < 0 ? 'down' : 'same');
        }

        $exchangeRate = 4100;

        $totalUsdThisMonth = Delivery::selectRaw('
                SUM(amount_usd + (amount_khr / ?)) as total_usd
            ', [$exchangeRate])
            ->where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_paid', true)
            ->whereNotNull('completed_at')
            ->whereMonth('completed_at', now()->month)
            ->whereYear('completed_at', now()->year)
            ->value('total_usd');

        $countCompleted = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->count();

        $countInStock = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::IN_STOCK->value)
            ->count();

        $countDelivery = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::ASSIGNED->value)
            ->count();
        $countReturn = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::RETURNED->value)
            ->count();

        $countCustomer = Customer::where('branch_id', $branchId)
            ->where('status', Status::ACTIVE->value)
            ->count();

        return view('pages.dashboard.index', compact('profit',
            'currentMonthTotal', 'lastMonthTotal',
            'percentageChange', 'trend', 'totalUsdThisMonth',
            'countCompleted',
            'countInStock',
            'countDelivery',
            'countReturn',
            'countCustomer'));
    }
}
