<?php

namespace App\Http\Controllers;

use App\Models\Bank;
use App\Models\Customer;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Validation\Rule;
use Yajra\DataTables\DataTables;

class CustomerController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:customers.view')->only(['index', 'show']);
        $this->middleware('permission:customers.create')->only(['create', 'store']);
        $this->middleware('permission:customers.edit')->only(['edit', 'update']);
        $this->middleware('permission:customers.delete')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $branchId = auth()->user()->default_branch_id;
            $data = Customer::where('branch_id', $branchId)
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->editColumn('created_at', function ($row) {
                    if (! $row->created_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->created_at)->format('d-F-Y');
                })
                ->addColumn('action', function ($row) {
                    return view('pages.customers.action-dropdown', compact('row'))->render();
                })
                ->rawColumns(['action'])
                ->make(true);
        }

        return view('pages.customers.index');
    }

    // Display Create Form
    public function create()
    {
        $banks = Bank::all();

        return view('pages.customers.create', compact('banks'));
    }

    public function edit($id)
    {
        $banks = Bank::all();
        $customer = Customer::with('banks')->findOrFail($id);

        $banksData = $customer->banks->map(function ($b) {
            return [
                'id' => $b->id,
                'bank' => $b->bank_id,
                'bank_account_number' => $b->bank_account_number,
                'bank_account_name' => $b->bank_account_name,
                'qr_codes' => $b->qr_code ? [[
                    'url' => asset('storage/'.$b->qr_code),
                    'name' => $b->qr_code,
                    'existing' => true,
                ]] : [],
            ];
        })->values()->toArray();

        return view('pages.customers.edit', compact('banks', 'customer', 'banksData'));
    }

    // Store New Customer
    public function store(Request $request)
    {
        $this->validateCustomer($request);
        $branchId = auth()->user()->default_branch_id;
        $request->merge(['branch_id' => $branchId, 'mou_price' => $request->mou_price ?? 0, 'user_id' => auth()->id()]);
        try {
            DB::beginTransaction();
            $customer = Customer::create($request->only(['customer_name', 'phone', 'type_of_customer', 'mou_price', 'telegram', 'address', 'branch_id', 'user_id', 'url_map']));
            $customer->ref_code = 'TID'.str_pad($customer->id, 7, '0', STR_PAD_LEFT);
            $customer->save();
            $this->saveBanks($request, $customer);

            DB::commit();

            return redirect()->route('customer.index')->with('success', 'Customer created successfully.');
        } catch (\Throwable $e) {
            DB::rollBack();

            return back()->with('error', 'Failed to create customer. '.$e->getMessage());
        }
    }

    // Update Customer
    public function update(Request $request, $id)
    {
        $this->validateCustomer($request, $id);
        $customer = Customer::findOrFail($id);
        $customer->update($request->only(['customer_name', 'phone', 'type_of_customer', 'mou_price', 'telegram', 'address', 'url_map']));

        // $customer->banks()->delete();
        foreach ($customer->banks as $bank) {
            $bank->delete(); // this now logs the deletion
        }
        $this->saveBanks($request, $customer);

        return redirect()->route('customer.index')->with('success', 'Customer updated successfully.');
    }

    private function validateCustomer(Request $request, $customerId = null)
    {
        $request->validate([
            'customer_name' => 'required|string|max:255',
            'phone' => [
                'required',
                'max:20',
                Rule::unique('customers', 'phone')->ignore($customerId),
            ],
            'type_of_customer' => 'required|in:Normal,VIP,VVIP',
            'mou_price' => 'nullable|numeric',
            'telegram' => 'nullable|string',
            'address' => 'nullable|string',
            'url_map' => 'nullable|string',
            'banks' => 'array',
            'banks.*.bank' => 'nullable|string',
            'banks.*.bank_account_number' => 'nullable|string',
            'banks.*.bank_account_name' => 'nullable|string',
            'banks.*.qr_codes.*' => 'nullable|image|mimes:jpeg,png,jpg|max:2048',
        ]);
    }

    private function saveBanks($request, $customer)
    {
        if (! is_array($request->banks)) {
            $customer->banks()->delete();

            return;
        }
        $request->validate([
            'banks' => ['nullable', 'array'],
            'banks.*.bank' => ['required_with:bank', 'integer', 'exists:bank,id'],
            'banks.*.bank_account_number' => ['required_with:bank', 'string'],
            'banks.*.bank_account_name' => ['required_with:bank', 'string'],
            'banks.*.qr_code' => ['nullable', 'file', 'image', 'max:2048'],
        ]);
        $existingBankIds = collect($request->banks)->pluck('id')->filter()->all();
        $customer->banks()->whereNotIn('id', $existingBankIds)->delete();

        foreach ($request->banks as $index => $bankData) {
            $qrPath = null;
            $bankRecord = null;

            if (! empty($bankData['id'])) {
                $bankRecord = $customer->banks()->find($bankData['id']);
            }

            if ($request->hasFile('banks.'.$index.'.qr_code')) {
                $file = $request->file('banks')[$index]['qr_code'];
                if ($file && $file->isValid()) {
                    $qrPath = $file->store('qr_codes', 'public');
                }
            }

            if (! $qrPath && ! empty($bankData['existing_qr_code'])) {
                $qrPath = $bankData['existing_qr_code'];
            }

            $dataToSave = [
                'bank_id' => $bankData['bank'],
                'bank_account_number' => $bankData['bank_account_number'],
                'bank_account_name' => $bankData['bank_account_name'],
                'qr_code' => $qrPath,
            ];

            if ($bankRecord) {
                $bankRecord->update($dataToSave);
            } else {
                $customer->banks()->create($dataToSave);
            }
        }
    }

    public function show($id)
    {
        $customer = Customer::with('banks')->findOrFail($id);

        return view('pages.customers.view', compact('customer'));
    }
}
