<?php

namespace App\Http\Controllers;

use App\Models\Branch;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Storage;
use Yajra\DataTables\DataTables;

class BranchController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:branch.view')->only(['index', 'show']);
        $this->middleware('permission:branch.create')->only(['create', 'store']);
        $this->middleware('permission:branch.edit')->only(['edit', 'update']);
        $this->middleware('permission:branch.delete')->only(['destroy']);
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Branch::orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('action', function ($row) {
                    return view('pages.configuration.branches.action-dropdown', compact('row'))->render();
                })
                ->make(true);
        }

        return view('pages.configuration.branches.index');
    }

    public function create()
    {
        return view('pages.configuration.branches.create');
    }

    public function store(Request $request)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_name' => 'nullable|string|max:255',
            'type' => 'nullable|string|max:255',
            'status' => 'nullable|string|max:50',
            'code' => 'nullable|string|max:50',
            'logo' => 'nullable|file|mimetypes:image/jpeg,image/png,image/gif,image/svg+xml|max:2048',
            'address' => 'nullable|string',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'url' => 'nullable|max:255',
        ]);

        if ($request->hasFile('logo')) {
            $validated['logo'] = $request->file('logo')->store('branch_logos', 'public');
        }

        Branch::create($validated);

        return redirect()->route('branch.index')->with('success', 'Branch created successfully.');
    }

    public function show(string $id)
    {
        $branch = Branch::findOrFail($id);

        return view('pages.configuration.branches.view', compact('branch'));
    }

    public function edit(string $id)
    {
        $branch = Branch::findOrFail($id);

        return view('pages.configuration.branches.edit', compact('branch'));
    }

    public function update(Request $request, string $id)
    {
        $validated = $request->validate([
            'name' => 'required|string|max:255',
            'short_name' => 'nullable|string|max:255',
            'type' => 'nullable|string|max:255',
            'status' => 'nullable|string|max:50',
            'code' => 'nullable|string|max:50',
            'logo' => 'nullable|file|mimetypes:image/jpeg,image/png,image/gif,image/svg+xml|max:2048',
            'address' => 'nullable|string',
            'phone' => 'nullable|string|max:20',
            'email' => 'nullable|email|max:255',
            'url' => 'nullable|max:255',
        ]);

        $branch = Branch::findOrFail($id);

        if ($request->hasFile('logo')) {
            $file = $request->file('logo');
            $filename = 'logo_'.time().'.'.$file->getClientOriginalExtension();

            // Store file and get the relative path
            $path = $file->storeAs('branch_logos', $filename, 'public');

            // Delete old logo if exists
            if ($branch->logo && Storage::disk('public')->exists($branch->logo)) {
                Storage::disk('public')->delete($branch->logo);
            }

            // Save relative path to DB (e.g. branch_logos/logo_123.jpg)
            $validated['logo'] = $path;
        }

        $branch->update($validated);

        return redirect()->route('branch.index')->with('success', 'Branch updated successfully.');
    }

    public function destroy(string $id)
    {
        $branch = Branch::findOrFail($id);
        $branch->delete();

        return response()->json(['success' => 'Branch deleted successfully.']);
    }

    public function switch(Request $request)
    {
        $request->validate([
            'branch_id' => 'required|exists:branches,id',
        ]);

        // Check if this branch belongs to the authenticated user
        if (! auth()->user()->branches->contains('id', $request->branch_id)) {
            return redirect()->back()->with('error', 'You do not have access to this branch.');
        }

        // Save to session
        session(['current_branch_id' => $request->branch_id]);
        auth()->user()->update([
            'default_branch_id' => $request->branch_id,
        ]);

        return redirect()->back();
    }
}
