<?php

namespace App\Http\Controllers;

use App\Enums\BookingStatus;
use App\Enums\ShippingMethod;
use App\Http\Requests\BookingOrder\CreateRequest;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\Warehouse;
use App\Models\ZonePrice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use SimpleSoftwareIO\QrCode\Facades\QrCode;

class BookingOrderController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        // $this->middleware('permission:delivery.view')->only(['index', 'show']);
        // $this->middleware('permission:delivery.create')->only(['create', 'store']);
        // $this->middleware('permission:delivery.edit')->only(['edit', 'update']);
        // $this->middleware('permission:delivery.delete')->only(['destroy']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        $currentBranch = auth()->user()->default_branch_id;
        $warehouse = Warehouse::get()->pluck('name', 'id');
        $customer = Customer::active()
            ->where('branch_id', $currentBranch)
            ->select('id', 'customer_name', 'phone', 'currency', 'type_of_customer', 'mou_price')
            ->get();

        $zone = ZonePrice::where('status', 'active')
            ->where('branch_id', $currentBranch)
            ->select('id', 'zone_name', 'price', 'currency')
            ->get();

        $shippingMethod = ShippingMethod::cases();

        return view('pages.booking.inquiry.create',
            compact('warehouse', 'customer', 'zone', 'shippingMethod')
        );
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(CreateRequest $request)
    {

        $item = $request->validated();

        DB::beginTransaction();

        try {
            $lastNumber = Delivery::getLastCodeNumber();

            foreach ($item['items'] as $item) {

                // generate running code
                $lastNumber++;
                $code = 'T'.str_pad($lastNumber, 7, '0', STR_PAD_LEFT);

                $localTracking = 'L'.collect(range(1, 12))
                    ->map(fn () => random_int(0, 9))
                    ->implode('');
                // upload photo (if exists)
                $photoPath = null;
                if (! empty($item['photo'])) {
                    $photoPath = $item['photo']->store('deliveries', 'public');
                }
                $customer = Customer::where('id', $request->customer_id)->first();
                $delivery = Delivery::create([
                    'branch_id' => auth()->user()->default_branch_id,

                    'from_warehouse_id' => $request->from_warehouse,
                    'to_warehouse_id' => $request->to_warehouse,

                    'shipping_method' => $item['shipping_method'],
                    'tracking_number' => $item['tracking_number'],
                    'local_tracking' => $localTracking,

                    'photo' => $photoPath,

                    'kg' => $item['kg'],
                    'pcs' => $item['pcs'],
                    'cbm' => $item['cbm'],

                    'code' => $code,
                    'customer_id' => $request->customer_id,
                    'zone_price_id' => null,

                    'receiver_address' => $customer->address ?? null,
                    'receiver_phone' => $customer->phone ?? null,

                    'delivery_price' => $item['amount_usd'] ?? 0,
                    'pickup_by' => null,

                    'amount_usd' => $item['amount_usd'] ?? 0,
                    'amount_khr' => 0,

                    'type_of_delivery' => 'Normal',
                    'delivery_by' => null,

                    'status' => BookingStatus::BOOKING->value,
                    'created_by' => auth()->id(),
                    'note' => null,
                    'pickup_at' => now(),
                ]);

                DeliveryStatusHistory::create([
                    'delivery_id' => $delivery->id,
                    'status' => BookingStatus::BOOKING->value,
                    'changed_by' => auth()->id(),
                    'note' => 'Booking created',
                ]);

            }

            DB::commit();

            return back()->with('success', 'Booking successfuly.');
            //     if (count($createdDeliveryIds) === 1) {
            //         return redirect()->route('inquiry.invoice.print', ['id' => $createdDeliveryIds[0]]);
            //     } else {
            //         return redirect()->route('inquiry.invoice.batch', ['ids' => implode(',', $createdDeliveryIds)]);

            //     }

        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Failed to create delivery batch. '.$e->getMessage());
        }
    }

    public function inquiryPrintInvoice($id)
    {
        $delivery = Delivery::with(['customer', 'statuses', 'payments'])->findOrFail($id);
        $tracking = route('tracking.search');
        $qrCodeBase64 = base64_encode(
            QrCode::format('png')->size(200)->generate($tracking.'?code='.$delivery->code)
        );

        return view('pages.delivery.inquiry.print-invoice', compact('delivery', 'qrCodeBase64'));
    }

    public function inquiryPrintBatchInvoice($ids)
    {
        $ids = explode(',', $ids);

        $deliveries = Delivery::with(['customer', 'statuses', 'payments'])
            ->whereIn('id', $ids)
            ->get();

        return view('pages.delivery.inquiry.print-batch-invoice', compact('deliveries'));
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
