<?php

namespace App\Http\Controllers;

use App\Enums\BookingStatus;
use App\Http\Requests\Booking\StatusRequest;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\Warehouse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Milon\Barcode\Facades\DNS1DFacade as DNS1D;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Yajra\DataTables\DataTables;

class BookingController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function bookingAjax(Request $request)
    {
        $branchId = $request->user()->default_branch_id;

        if ($request->ajax()) {
            $data = Delivery::where('branch_id', $branchId)
                ->whereIn('status', [
                    BookingStatus::BOOKING->value,
                    BookingStatus::IN_TRANSIT->value,
                    BookingStatus::ARRIVEDWAREHOUSE->value,
                    BookingStatus::ARRIVEDWAREHOUSEPP->value])
                ->when($request->from_warehouse, function ($query, $from_warehouse) {
                    return $query->where('from_warehouse_id', $from_warehouse);
                })
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->status, function ($query, $status) {
                    return $query->where('status', $status);
                })
                ->when($request->entry_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->entry_at);

                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('created_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        return $query->whereBetween('created_at', [
                            $dates[0].' 00:00:00',
                            $dates[0].' 23:59:59',
                        ]);
                    }

                    return $query;
                })

                ->when($request->completed_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->completed_at);

                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('completed_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        $date = $dates[0];

                        return $query->whereBetween('completed_at', [
                            $date.' 00:00:00',
                            $date.' 23:59:59',
                        ]);
                    }

                    return $query;
                })

                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('ref_code', function ($row) {
                    return $row->customer->ref_code ?? 'N/A';
                })
                ->addColumn('photo', function ($row) {
                    return $row->photo ? asset('storage/'.$row->photo) : null;
                })

                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('from_warehouse_name', function ($row) {
                    return $row->fromWarehouse->name ?? 'N/A';
                })
                ->addColumn('to_warehouse_name', function ($row) {
                    return $row->toWarehouse->name ?? 'N/A';
                })

                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })
                ->addColumn('truck_by', function ($row) {
                    return $row->truckBy->display_name ?? 'N/A';
                })
                ->addColumn('created_by', function ($row) {
                    return $row->createdBy->display_name ?? 'N/A';
                })
                ->editColumn('created_at', function ($row) {
                    if (! $row->created_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->created_at)->format('d-F-Y');
                })
                ->addColumn('status_label', function ($row) {
                    $status = BookingStatus::tryFrom($row->status);
                    $label = $status?->label() ?? 'Unknown';
                    $colorClass = match ($status) {
                        BookingStatus::BOOKING => 'text-warning',
                        BookingStatus::ARRIVEDWAREHOUSE => 'text-info',
                        BookingStatus::IN_TRANSIT => 'text-success',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                    HTML;
                })

                ->addColumn('action', function ($row) {
                    return view('pages.booking.action-dropdown', [
                        'row' => $row,
                        'canEdit' => $row->is_paid !== true,
                    ])->render();
                })

                ->addColumn('status_cancel', function ($row) {
                    $hasReason = $row->cancel_reason_id !== null;
                    $reason = $hasReason ? 'Yes' : 'No';
                    $colorClass = $hasReason ? 'text-error' : 'text-success';
                    $tooltipText = $hasReason ? $row->cancelReason->reason : 'No cancel reason';

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}" x-tooltip.delay.500="'{$tooltipText}'">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$reason}</span>
                        </div>
                    HTML;
                })

                ->rawColumns(['action', 'status_label', 'status_cancel'])
                ->make(true);

        }
    }

    public function index(Request $request)
    {
        $branchId = $request->user()->default_branch_id;
        $warehouse = Warehouse::get()->pluck('name', 'id');
        $customer = Customer::active()
            ->where('branch_id', $branchId)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();

        $statusOptions = collect(BookingStatus::cases())->map(function ($status) {
            return [
                'value' => $status->value,
                'label' => $status->label(),
            ];
        });

        return view('pages.booking.index', compact('warehouse', 'customer', 'statusOptions'));
    }

    /* ajx */
    public function processUpdateStatus(StatusRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $delivery = Delivery::findOrFail($data['delivery_id']);
            $delivery->status = $data['status'];
            $delivery->note = $data['note'];
            $delivery->save();

            DeliveryStatusHistory::create([
                'delivery_id' => $data['delivery_id'],
                'status' => $delivery->status,
                'note' => $delivery->note ?? $delivery->status,
                'changed_by' => auth()->user()->id,
                'changed_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Update status successfully.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('updated booking status error: '.$e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while processing. Please try again later.',
            ], 500);
        }
    }

    /**printInvoice */
    public function printInvoice(string $id)
    {
        $delivery = Delivery::with(['fromWarehouse', 'toWarehouse'])->findOrFail($id);
        $tracking = route('home.index').'?tracking='.$delivery->local_tracking;

        $qrCodeBase64 = base64_encode(
            QrCode::format('png')->size(200)->generate($tracking)
        );
        // 1D Barcode (Code128)
        $barcode1D = DNS1D::getBarcodePNG($delivery->code, 'C128');

        return view('pages.booking.inquiry.print-invoice', compact('delivery', 'qrCodeBase64', 'barcode1D'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        //
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, string $id)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }
}
