<?php

namespace App\Imports;

use App\Enums\DeliveryStatus;
use App\Enums\Status;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use Illuminate\Support\Collection;
use Maatwebsite\Excel\Concerns\ToCollection;
use Maatwebsite\Excel\Concerns\WithHeadingRow;

class DeliveryImport implements ToCollection, WithHeadingRow
{
    protected ?int $pickupBy;

    protected int $lastNumber;

    public function __construct(
        ?int $pickupBy,
    ) {
        $this->pickupBy = $pickupBy;
        $this->lastNumber = Delivery::getLastCodeNumber();
    }

    public function collection(Collection $rows)
    {
        foreach ($rows as $row) {
            $this->lastNumber++;

            $code = 'T'.str_pad($this->lastNumber, 7, '0', STR_PAD_LEFT);

            $customer = Customer::firstOrCreate(
                ['phone' => $row['phone']], // match by phone
                [
                    'branch_id' => auth()->user()->default_branch_id,
                    'user_id' => auth()->id(),
                    'customer_name' => $row['phone'] ?? 'Unknown Customer',
                    'address' => $row['address'] ?? null,
                    'status' => Status::ACTIVE->value,
                    'type_of_customer' => 'Normal',
                    'currency' => $row['currency'] ?? 'USD',
                    'first_delivery_date' => now(),
                ]
            );

            // Update ref_code if not set
            if (! $customer->ref_code) {
                $customer->ref_code = 'TID'.str_pad($customer->id, 7, '0', STR_PAD_LEFT);
                $customer->save();
            }

            $delivery = Delivery::create([
                'branch_id' => auth()->user()->default_branch_id,
                'code' => $code,
                'customer_id' => $customer->id,

                // Excel columns
                'zone_price_id' => $row['zone_id'] ?? null,
                'receiver_address' => $row['address'],
                'receiver_phone' => $row['phone'],
                'delivery_price' => $row['delivery_price'] ?? 0,
                'amount_usd' => $row['amount_usd'] ?? 0,
                'amount_khr' => $row['amount_khr'] ?? 0,

                // Fixed / request data
                'pickup_by' => $this->pickupBy,
                'delivery_by' => null,
                'type_of_delivery' => 'Normal',
                'status' => DeliveryStatus::IN_STOCK->value,
                'created_by' => auth()->id(),
                'note' => null,
                'pickup_at' => now(),
            ]);

            DeliveryStatusHistory::create([
                'delivery_id' => $delivery->id,
                'status' => DeliveryStatus::IN_STOCK->value,
                'changed_by' => auth()->id(),
                'note' => 'Import by '.auth()->user()->display_name,
            ]);

        }
    }
}
