<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Enums\Status;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\SummaryDriverPayment;
use App\Models\SummaryShopPayment;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class ShopSummaryPaymentController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:customer-payment.transfer')->only(['transferToShopCompleted']);
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = SummaryShopPayment::with('confirmPayments', 'customer')

             ->when($request->customer_id, function ($query, $customer_id) {
                return $query->where('customer_id', $customer_id);
            })
            ->when($request->status, function ($query, $status) {
                return $query->where('status', $status);
            })
            ->when($request->entry_at, function ($query) use ($request) {
                $dates = explode(' to ', $request->entry_at);
                if (count($dates) === 2) {
                    $from = $dates[0].' 00:00:00';
                    $to = $dates[1].' 23:59:59';

                    return $query->whereBetween('created_at', [$from, $to]);
                } elseif (count($dates) === 1 && $dates[0]) {
                    return $query->whereBetween('created_at', [
                        $dates[0].' 00:00:00',
                        $dates[0].' 23:59:59',
                    ]);
                }

                return $query;
            })
             ->when($request->completed_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->completed_at);
                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('payment_date', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        return $query->whereBetween('payment_date', [
                            $dates[0].' 00:00:00',
                            $dates[0].' 23:59:59',
                        ]);
                    }

                    return $query;
                })
            ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('customer', function ($row) {
                    $customer = $row->customer;
                    $name = $customer->customer_name ?? '-';
                    $phone = $customer->phone ?? '-';

                    return '
                        <div class="flex items-center space-x-3">
                            <div class="flex flex-col">
                                <span class="font-medium text-slate-700 dark:text-navy-100">'.$name.'</span>
                                <span class="text-xs text-slate-400 dark:text-navy-300">'.$phone.'</span>
                            </div>
                        </div>
                    ';
                })

                ->addColumn('cod', function ($row) {
                    return '
                        <div class="flex flex-col text-sm">
                         <a class="text-success">Payment to customer</a>
                            <ul class="list-disc list-inside mt-1 text-slate-600 dark:text-navy-200">
                                <li>'.number_format($row->total_amount_usd, 2).' USD</li>
                                <li>'.number_format($row->total_amount_khr, 2).' KHR</li>
                            <ul>
                        </div>
                    ';
                })

                ->addColumn('collect_amount', function ($row) {
                    $amountUSD = $row->confirmPayments->where('currency', 'usd')->sum('amount');
                    $paymentMethodUSD = optional($row->confirmPayments->where('currency', 'usd')->first())->payment_method ?? 'N/A';
                    $amountKHR = $row->confirmPayments->where('currency', 'khr')->sum('amount');
                    $paymentMethodKHR = optional($row->confirmPayments->where('currency', 'khr')->first())->payment_method ?? 'N/A';

                    return '
                        <div class="flex flex-col text-sm">
                         <a class="text-success">Payment to customer</a>
                            <ul class="list-disc list-inside mt-1 text-slate-600 dark:text-navy-200">
                                <li>'.number_format($amountUSD, 2).' USD ('.ucfirst($paymentMethodUSD).')</li>
                                <li>'.number_format($amountKHR, 2).' KHR ('.ucfirst($paymentMethodKHR).')</li>
                            <ul>
                        </div>
                    ';
                })

                ->addColumn('created_by', function ($row) {
                    return $row->createdBy->display_name ?? 'N/A';
                })
                ->addColumn('updated_by', function ($row) {
                    return $row->updatedBy->display_name ?? 'N/A';
                })
                ->editColumn('payment_date', function ($row) {
                    if (! $row->payment_date) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->payment_date)->format('d/m/Y h:i A');
                })

                ->addColumn('action', function ($row) {
                    return view('pages.summary-shop.action-dropdown', compact('row'))->render();
                })
                ->addColumn('status_label', function ($row) {
                    $status = Status::tryFrom($row->status);
                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        Status::PENDING => 'text-error dark:text-error-100',
                        Status::PAID => 'text-success',
                        Status::REJECTED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                            <div class="border badge space-x-2.5 {$colorClass}">
                                <div class="size-2 rounded-full bg-current"></div>
                                <span>{$label}</span>
                            </div>
                        HTML;
                })
                ->addColumn('status_clickable', function ($row) {
                    return Status::tryFrom($row->status) === Status::PENDING;
                })
                ->addColumn('paymentData', function ($data) {
                    return [
                        'invoiceNumber' => $data->invoice_number,
                        'deliveryFee' => $data->service_fee,
                        'customerName' => $data->customer->customer_name,
                        'amountUsd' => (float) $data->total_amount_usd,
                        'amountKhr' => (float) $data->total_amount_khr,
                        'note' => $data->note,
                        'payments' => $data->confirmPayments->map(function ($p) {
                            return [
                                'id' => $p->id,
                                'amount' => (float) $p->amount,
                                'currency' => $p->currency,
                                'payment_method' => $p->payment_method,
                            ];
                        })->values()->all(),
                    ];
                })
                ->rawColumns(['action', 'status_label', 'status_clickable', 'paymentData', 'collect_amount', 'cod', 'customer'])
                ->make(true);

        }
        $branchId = $request->user()->default_branch_id;
        $customer = Customer::active()
            ->where('branch_id', $branchId)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();

       $statusOptions = collect(Status::cases())
        ->filter(fn($status) => in_array($status->value, ['paid', 'pending']))
        ->map(fn($status) => [
            'value' => $status->value,
            'label' => $status->label(),
        ]);

        
        return view('pages.summary-shop.index',compact('customer','statusOptions'));
    }

    public function transferToShopCompleted(Request $request)
    {

        $request->validate([
            'invoice_number' => 'required|exists:summary_shop_payment,invoice_number',
            'payments' => 'required|array|min:1',
            'payments.*.amount' => 'required|numeric',
            'payments.*.id' => 'required|numeric',
            'payments.*.currency' => 'required|in:usd,khr',
            'payments.*.payment_method' => 'required|in:cash,transfer',
        ]);

        $summary = SummaryShopPayment::where('invoice_number', $request->invoice_number)->firstOrFail();

        try {
            DB::beginTransaction();

            $summary->update([
                'payment_date' => now(),
                'status' => Status::PAID->value,
                'note' => $request->note,
                'updated_by' => auth()->id(),
            ]);

            foreach ($request->payments as $payment) {
                $confirmPayment = $summary->confirmPayments()
                    ->where('id', $payment['id'])
                    ->first();

                if ($confirmPayment) {
                    $confirmPayment->update([
                        'amount' => $payment['amount'],
                        'currency' => $payment['currency'],
                        'payment_method' => $payment['payment_method'],
                    ]);
                }
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Transfer payment was successful.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong. Please try again later.',
            ], 500);
        }
    }

    public function eod(Request $request)
    {

        // Parse date filter
        if ($request->filled('date')) {
            $dates = explode(' to ', $request->date);

            $start = Carbon::parse($dates[0])->startOfDay();
            $end = isset($dates[1])
                ? Carbon::parse($dates[1])->endOfDay()
                : Carbon::parse($dates[0])->endOfDay();
        } else {
            $start = now()->startOfDay();
            $end = now()->endOfDay();
        }

        $summary = Delivery::selectRaw("
                SUM(CASE WHEN status='completed' AND completed_at BETWEEN ? AND ? THEN 1 ELSE 0 END) AS completed,
                SUM(CASE WHEN status='canceled' AND canceled_at BETWEEN ? AND ? THEN 1 ELSE 0 END) AS canceled,
                SUM(CASE WHEN status='assigned' AND assigned_at BETWEEN ? AND ? THEN 1 ELSE 0 END) AS assigned,
                SUM(CASE WHEN status='in_stock' AND created_at BETWEEN ? AND ? THEN 1 ELSE 0 END) AS in_stock,
                 SUM(CASE 
                    WHEN status='completed' 
                    AND completed_at BETWEEN ? AND ? 
                    THEN amount_usd ELSE 0 END) AS total_usd,

                SUM(CASE 
                    WHEN status='completed' 
                    AND completed_at BETWEEN ? AND ? 
                    THEN amount_khr ELSE 0 END) AS total_khr
            ", [
            $start, $end,
            $start, $end,
            $start, $end,
            $start, $end,
            $start, $end,
            $start, $end,
        ])->first();

        // Fetch all SummaryShopPayments for today with their confirmPayments
        $data = SummaryShopPayment::with('confirmPayments', 'customer')
            ->whereBetween('payment_date', [$start->toDateString(), $end->toDateString()])
            ->get();

        $grandTotal = [
            'usd' => $data->sum('total_amount_usd'),
            'khr' => $data->sum('total_amount_khr'),

            // Sum from related confirmPayments
            'paid_usd' => $data->flatMap(fn ($data) => $data->confirmPayments)
                ->where('currency', 'usd')
                ->sum('amount'),

            'paid_khr' => $data->flatMap(fn ($data) => $data->confirmPayments)
                ->where('currency', 'khr')
                ->sum('amount'),
            // delivery
            'total_usd' => $summary->total_usd,
            'total_khr' => $summary->total_khr,
            'total_package' => ($summary->completed + $summary->assigned + $summary->canceled + $summary->in_stock),
            'service_fee' => $data->sum('service_fee'),
            'completed' => $summary->completed,
            'assigned' => $summary->assigned,
            'canceled' => $summary->canceled,
            'returned' => 0,
            'in_stock' => $summary->in_stock,
            'total_shop' => $data->pluck('customer_id')->unique()->count(),
        ];

        $dailySummaryDriver = SummaryDriverPayment::whereBetween(
            'payment_date',
            [$start->toDateString(), $end->toDateString()]
        )
            ->selectRaw('
            SUM(total_amount_usd) as total_amount_usd,
            SUM(total_amount_khr) as total_amount_khr,
            SUM(driver_collect_usd) as driver_collect_usd,
            SUM(driver_collect_khr) as driver_collect_khr,
            COUNT(DISTINCT driver_id) as total_driver
        ')
            ->first();

        return view('pages.summary-shop.eod', compact(
            'data',
            'grandTotal',
            'dailySummaryDriver'
        ));
    }
}
