<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Enums\Status;
use App\Models\Branch;
use App\Models\ConfirmShopPayment;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\ShopCollectDeliveryPayment;
use App\Models\SummaryShopPayment;
use App\Models\User;
use App\Models\ZonePrice;
use Barryvdh\DomPDF\Facade\Pdf;
use Carbon\Carbon;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class ShopCollectPaymentController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:customer-payment.view')->only(['index', 'show', 'all']);
        $this->middleware('permission:customer-payment.create')->only(['create', 'store']);
        $this->middleware('permission:customer-payment.edit')->only(['edit', 'update']);
        $this->middleware('permission:customer-payment.delete')->only(['destroy']);
        $this->middleware('permission:customer-payment.download')->only(['download']);
    }

    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->where('status', DeliveryStatus::COMPLETED->value)
                ->where('is_collected', true)
                ->where('is_paid', false)
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })

                ->addColumn('delivery_by', function ($row) {
                    return $row->deliveryBy->display_name ?? 'N/A';
                })
                ->editColumn('pickup_at', function ($row) {
                    if (! $row->pickup_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->pickup_at)->format('d/m/Y h:i A');
                })
                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);

                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                        HTML;
                })
                ->rawColumns(['status_label'])
                ->make(true);

        }

        $deliveryBy = User::get()->pluck('name', 'id');
        $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();

        $zone = ZonePrice::where('status', 'active')
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'zone_name', 'price', 'currency')
            ->get();

        return view('pages.shop-collect-payment.index',
            compact('deliveryBy', 'customer', 'zone')
        );
    }

    public function all(Request $request)
    {
        if ($request->ajax()) {

            $data = Delivery::leftJoin('delivery_payments', 'delivery_payments.delivery_id', '=', 'delivery.id')
                ->selectRaw('
                    customer_id,
                    SUM(CASE WHEN delivery.status = ? THEN delivery.amount_usd ELSE 0 END) as total_usd,
                    SUM(CASE WHEN delivery.status = ? THEN delivery.amount_khr ELSE 0 END) as total_khr,
                    SUM(CASE WHEN delivery.status = ? THEN delivery.delivery_price ELSE 0 END) as total_delivery_price,
                    SUM(CASE WHEN delivery_payments.payment_to IN ("driver","company") AND delivery_payments.currency = "usd" THEN delivery_payments.amount ELSE 0 END) as driver_usd,
                    SUM(CASE WHEN delivery_payments.payment_to IN ("driver","company") AND delivery_payments.currency = "khr" THEN delivery_payments.amount ELSE 0 END) as driver_khr,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as completed,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as in_stock,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as assigned,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as canceled
                ', [
                    DeliveryStatus::COMPLETED->value,
                    DeliveryStatus::COMPLETED->value,
                    DeliveryStatus::COMPLETED->value,
                    DeliveryStatus::COMPLETED->value,
                    DeliveryStatus::IN_STOCK->value,
                    DeliveryStatus::ASSIGNED->value,
                    DeliveryStatus::CANCELED->value,
                ])
                 ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->where('delivery.branch_id', auth()->user()->default_branch_id)
                ->where('delivery.is_paid', false)
                ->groupBy('delivery.customer_id')
                ->having('completed', '>', 0);

            return DataTables::of($data)
                ->addColumn('customers', function ($row) {
                    return '
                        <div>
                            <div><h3>'.$row->customer?->customer_name.'</h3></div>
                            <div><strong>'.$row->customer->phone.'</strong> </div>
                            <div><strong>'.$row->customer->ref_code.'</strong> </div>
                            <hr/>
                            <div class="mt-2 badge p-2 space-x-2 bg-blue-100 text-blue-800 dark:bg-blue-500 dark:text-white"><strong><a href="'.route('shop-collect-payment.show', $row->customer_id).'">Detail</a></strong></div>
                        </div>
                    ';
                })
                ->addColumn('real_amount', function ($row) {
                    $usd = (float) $row->total_usd;
                    $khr = (float) $row->total_khr;

                    return '
                        <div>
                            <div><strong>USD:</strong> '.$usd.'</div>
                            <div><strong>KHR:</strong> '.$khr.'</div>
                            
                        </div>
                    ';
                })
                ->addColumn('total_amount', function ($row) {

                    $usd = (float) $row->driver_usd;
                    $khr = (float) $row->driver_khr;
                    $delivery = (float) $row->total_delivery_price;

                    $paidUsd = 0;
                    $paidKhr = 0;

                    // Case 1: USD can cover delivery fee
                    if ($usd >= $delivery) {
                        $paidUsd = $usd - $delivery;
                        $paidKhr = $khr;

                        // Case 2: USD cannot cover delivery fee
                    } else {
                        $remainingDelivery = $delivery - $usd;
                        $khrInUsd = $khr / getExchangeRate();

                        if ($khrInUsd >= $remainingDelivery) {
                            // KHR covers remaining delivery
                            $usedKhr = $remainingDelivery * getExchangeRate();
                            $paidUsd = 0;
                            $paidKhr = $khr - $usedKhr;
                        }
                    }

                    // Final display
                    $paidDisplay = ($paidUsd > 0 || $paidKhr > 0)
                        ? number_format($paidUsd, 2).' | '.number_format($paidKhr)
                        : '0';

                    return '
                        <div>
                            <div><strong>USD:</strong> '.$usd.'</div>
                            <div><strong>KHR:</strong> '.$khr.'</div>
                            <div><strong>Delivery Fee:</strong> '.$delivery.'</div>
                            <hr/>
                            <div class="mt-2 badge p-2 bg-blue-100 text-blue-800 dark:bg-blue-500 dark:text-white">
                                <strong>Paid to customer:</strong> '.$paidDisplay.'
                            </div>
                        </div>
                    ';
                })

                ->addColumn('status_summary', function ($row) {
                    return '
                    <ul class="text-sm">
                        <li class="text-success">✔ Completed: '.$row->completed.'</li>
                        <li class="text-info">📦 In Stock: '.$row->in_stock.'</li>
                        <li class="text-primary">🚚 Assigned: '.$row->assigned.'</li>
                        <li class="text-error">✖ Canceled: '.$row->canceled.'</li>
                    </ul>
                ';
                })
                ->rawColumns(['total_amount', 'status_summary', 'customers', 'real_amount'])
                ->make(true);
        }
        $today = Carbon::today();
    
         $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->withCount(['deliveries as completed_today_count' => function ($query) use ($today) {
                $query->where('status', 'completed')
                ->orWhere('is_collected', true)
                ->Where('is_paid',false)
                ->whereDate('completed_at', $today);
            }])
            ->having('completed_today_count', '>', 0)
            ->get();

        return view('pages.shop-collect-payment.all',
            compact('customer')
        );
    }

    public function generatedInvoice($customerId)
    {

        $payments = Delivery::leftJoin('delivery_payments', 'delivery_payments.delivery_id', '=', 'delivery.id')
            ->selectRaw('
                    customer_id,
                    SUM(CASE WHEN delivery.status = ? THEN delivery.amount_usd ELSE 0 END) as total_usd,
                    SUM(CASE WHEN delivery.status = ? THEN delivery.amount_khr ELSE 0 END) as total_khr,
                    SUM(CASE WHEN delivery.status = ? THEN delivery.delivery_price ELSE 0 END) as total_delivery_price,
                    SUM(CASE WHEN delivery_payments.payment_to IN ("driver","company") AND delivery_payments.currency = "usd" THEN delivery_payments.amount ELSE 0 END) as driver_usd,
                    SUM(CASE WHEN delivery_payments.payment_to IN ("driver","company") AND delivery_payments.currency = "khr" THEN delivery_payments.amount ELSE 0 END) as driver_khr,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as completed,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as in_stock,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as assigned,
                    COUNT(CASE WHEN delivery.status = ? THEN 1 END) as canceled
                ', [
                DeliveryStatus::COMPLETED->value,
                DeliveryStatus::COMPLETED->value,
                DeliveryStatus::COMPLETED->value,
                DeliveryStatus::COMPLETED->value,
                DeliveryStatus::IN_STOCK->value,
                DeliveryStatus::ASSIGNED->value,
                DeliveryStatus::CANCELED->value,
            ])
            ->where('delivery.branch_id', auth()->user()->default_branch_id)
            ->where('delivery.is_paid', false)
            ->where('delivery.status', DeliveryStatus::COMPLETED->value)
            ->where('delivery.is_collected', 1)
            ->where('delivery.customer_id', $customerId)
            ->groupBy('delivery.customer_id')
            ->first();

        $deliveryIds = Delivery::where('branch_id', auth()->user()->default_branch_id)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_collected', true)
            ->where('is_paid', false)
            ->where('customer_id', $customerId)
            ->pluck('id')
            ->toArray();

        if (empty($deliveryIds)) {
            return response()->json([
                'success' => false,
                'message' => 'Driver not yet clear payment, please clear payment with driver first.',
            ], 422);
        }

        try {
            DB::beginTransaction();

            $serviceFee = $payments->total_delivery_price;
            $totalAmountUsd = $payments->driver_usd;
            $totalAmountKhr = $payments->driver_khr;

            $today = Carbon::today();

            $newPackageCount = Delivery::where('customer_id', $customerId)
                ->whereDate('created_at', $today)
                ->where('is_paid', 0)
                ->count();

            $oldPackageCount = Delivery::where('customer_id', $customerId)
                ->whereDate('created_at', '<', $today)
                ->where('status', '!=', 'completed')
                ->where('is_paid', 0)
                ->count();

            $statusCounts = Delivery::where('customer_id', $customerId)
                ->where('is_paid', 0)
                ->selectRaw('status, COUNT(*) as total')
                ->groupBy('status')
                ->pluck('total', 'status');

            // Create Payment Summary
            $summary = SummaryShopPayment::create([
                'customer_id' => $customerId,
                'invoice_number' => 'INV-'.now()->format('YmdHis'),
                'exchange_rate' => getExchangeRate(),
                'total_amount_usd' => $totalAmountUsd,
                'total_amount_khr' => $totalAmountKhr,
                'service_fee' => $serviceFee,
                'status' => Status::PENDING->value,
                'total_new_package' => $newPackageCount ?? 0,
                'total_old_package' => $oldPackageCount ?? 0,
                'completed' => $statusCounts['completed'] ?? 0,
                'assigned' => $statusCounts['assigned'] ?? 0,
                'canceled' => $statusCounts['canceled'] ?? 0,
                'return' => $statusCounts['return'] ?? 0,
                'in_stock' => $statusCounts['in_stock'] ?? 0,
                'created_by' => auth()->id(),
            ]);

            ConfirmShopPayment::create([
                'summary_shop_payment_id' => $summary->id,
                'customer_id' => $customerId,
                'amount' => $totalAmountUsd - $serviceFee,
                'currency' => 'usd',
            ]);

            ConfirmShopPayment::create([
                'summary_shop_payment_id' => $summary->id,
                'customer_id' => $customerId,
                'amount' => $totalAmountKhr,
                'currency' => 'khr',
            ]);

            // delivery complted
            foreach ($deliveryIds as $deliveryId) {
                ShopCollectDeliveryPayment::create([
                    'summary_shop_payment_id' => $summary->id,
                    'customer_id' => $customerId,
                    'delivery_id' => $deliveryId,
                    'status' => 'collected', // or reverted
                    'payment_date' => now(),
                    'note' => 'collected payment with shop',
                    'created_by' => auth()->id(),
                ]);

            }

            $assignCancel = Delivery::where('customer_id', $customerId)
                ->where('is_paid', 0)
                ->whereIn('status', [
                    DeliveryStatus::CANCELED->value,
                    DeliveryStatus::IN_STOCK->value,
                    DeliveryStatus::ASSIGNED->value,
                ])
                ->select(['id', 'status'])
                ->get();

            // delivery assign & canceled
            foreach ($assignCancel as $del) {
                ShopCollectDeliveryPayment::create([
                    'summary_shop_payment_id' => $summary->id,
                    'customer_id' => $customerId,
                    'delivery_id' => $del->id,
                    'status' => $del->status,
                    'payment_date' => now(),
                    'note' => 'assigned & canceled recorde',
                    'created_by' => auth()->id(),
                ]);
                Delivery::where('id', $del->id)->where('status', DeliveryStatus::CANCELED->value)->update([
                    'is_paid' => 1,
                ]);
            }

            Delivery::whereIn('id', $deliveryIds)->update([
                'is_paid' => 1,
            ]);

            sendTelegramReportNotification([
                'invoice_number' => $summary->invoice_number,
                'customer' => $summary->customer?->customer_name,
                'total_amount_usd' => $summary->total_amount_usd,
                'total_amount_khr' => $summary->total_amount_khr,
                'total_new_package' => $summary->total_new_package,
                'total_old_package' => $summary->total_old_package,
                'completed' => $summary->completed,
                'assigned' => $summary->assigned,
                'canceled' => $summary->canceled,
                'return' => $summary->return,
                'in_stock' => $summary->in_stock,
                'service_fee' => $summary->service_fee,
                'date' => $summary->created_at,
            ]);
            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Invoice generated',
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();

            return response()->json([
                'success' => false,
                'message' => 'Invoice generated',
            ], 422);

        }
    }

    public function show(Request $request, $id)
    {
        $customerId = $request->customer_id ?? $id;

        $deliveries = Delivery::with('deliveryBy', 'payments')->where('customer_id', $customerId)
            // ->whereIn('id', $request->delivery_ids)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_collected', true)
            ->where('is_paid', false)
            ->get();

        $statusCounts = Delivery::where('customer_id', $customerId)
            ->where('is_paid', 0)
            ->selectRaw('status, COUNT(*) as total')
            ->groupBy('status')
            ->pluck('total', 'status');

        if ($deliveries->isEmpty()) {
            return redirect()->back()->with('error', 'No deliveries found for the selected customer.');
        }

        $customer = Customer::findOrFail($customerId);

        $branch = Branch::findOrFail(auth()->user()->default_branch_id);

        return view('pages.shop-collect-payment.view-collect-payment', [
            'deliveries' => $deliveries,
            'customer' => $customer,
            'exchangeRate' => getExchangeRate(),
            'invoiceNumber' => 'INV-NOTYETGENERATED',
            'branch' => $branch,
            'statusCounts' => $statusCounts,
        ]);
    }

    public function store(Request $request)
    {

        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'delivery_ids' => 'required|array',
            'delivery_ids.*' => 'exists:delivery,id',
            'payments' => 'required|array|min:1',
            'payments.*.amount' => 'required|numeric',
            'payments.*.currency' => 'required|in:usd,khr',
            'service_fee' => 'required|min:0',
            // 'payments.*.payment_method' => 'required|in:cash,transfer',
        ]);

        try {
            DB::beginTransaction();

            $deliveryIds = $request->delivery_ids;
            $customerId = $request->customer_id;
            $serviceFee = $request->service_fee;
            $totalAmountUsd = $request->total_amount_usd;
            $totalAmountKhr = $request->total_amount_khr;

            $today = Carbon::today();

            $newPackageCount = Delivery::where('customer_id', $request->customer_id)
                ->whereDate('created_at', $today)
                ->where('is_paid', 0)
                ->count();

            $oldPackageCount = Delivery::where('customer_id', $request->customer_id)
                ->whereDate('created_at', '<', $today)
                ->where('status', '!=', 'completed')
                ->where('is_paid', 0)
                ->count();

            $statusCounts = Delivery::where('customer_id', $request->customer_id)
                ->where('is_paid', 0)
                ->selectRaw('status, COUNT(*) as total')
                ->groupBy('status')
                ->pluck('total', 'status');

            // Create Payment Summary
            $summary = SummaryShopPayment::create([
                'customer_id' => $customerId,
                'invoice_number' => 'INV-'.now()->format('YmdHis'),
                'exchange_rate' => getExchangeRate(),
                'total_amount_usd' => $totalAmountUsd,
                'total_amount_khr' => $totalAmountKhr,
                'service_fee' => $serviceFee,
                'status' => Status::PENDING->value,
                'total_new_package' => $newPackageCount ?? 0,
                'total_old_package' => $oldPackageCount ?? 0,
                'completed' => $statusCounts['completed'] ?? 0,
                'assigned' => $statusCounts['assigned'] ?? 0,
                'canceled' => $statusCounts['canceled'] ?? 0,
                'return' => $statusCounts['return'] ?? 0,
                'in_stock' => $statusCounts['in_stock'] ?? 0,

                // 'payment_date' => now(),
                'created_by' => auth()->id(),
            ]);

            $totalUsd = 0;
            $totalKhr = 0;

            foreach ($request->payments as $payment) {
                ConfirmShopPayment::create([
                    'summary_shop_payment_id' => $summary->id,
                    'customer_id' => $customerId,
                    'amount' => $payment['amount'],
                    'currency' => $payment['currency'],
                    // 'payment_method' => $payment['payment_method'],
                ]);
                if ($payment['currency'] === 'usd') {
                    $totalUsd += $payment['amount'];
                } else {
                    $totalKhr += $payment['amount'];
                }

            }

            // delivery complted
            foreach ($deliveryIds as $deliveryId) {
                ShopCollectDeliveryPayment::create([
                    'summary_shop_payment_id' => $summary->id,
                    'customer_id' => $customerId,
                    'delivery_id' => $deliveryId,
                    'status' => 'collected', // or reverted
                    'payment_date' => now(),
                    'note' => 'collected payment with shop',
                    'created_by' => auth()->id(),
                ]);

            }

            $assignCancel = Delivery::where('customer_id', $request->customer_id)
                ->where('is_paid', 0)
                ->whereIn('status', [
                    DeliveryStatus::CANCELED->value,
                    DeliveryStatus::ASSIGNED->value,
                    DeliveryStatus::IN_STOCK->value,
                ])
                ->select(['id', 'status'])
                ->get();

            // delivery assign & canceled
            foreach ($assignCancel as $del) {
                ShopCollectDeliveryPayment::create([
                    'summary_shop_payment_id' => $summary->id,
                    'customer_id' => $customerId,
                    'delivery_id' => $del->id,
                    'status' => $del->status,
                    'payment_date' => now(),
                    'note' => 'assigned & canceled recorde',
                    'created_by' => auth()->id(),
                ]);
                Delivery::where('id', $del->id)->where('status', DeliveryStatus::CANCELED->value)->update([
                    'is_paid' => 1,
                ]);
            }

            Delivery::whereIn('id', $deliveryIds)->update([
                'is_paid' => 1,
            ]);

            sendTelegramReportNotification([
                'invoice_number' => $summary->invoice_number,
                'customer' => $summary->customer?->customer_name,
                'total_amount_usd' => $summary->total_amount_usd,
                'total_amount_khr' => $summary->total_amount_khr,
                'total_new_package' => $summary->total_new_package,
                'total_old_package' => $summary->total_old_package,
                'completed' => $summary->completed,
                'assigned' => $summary->assigned,
                'canceled' => $summary->canceled,
                'return' => $summary->return,
                'in_stock' => $summary->in_stock,
                'service_fee' => $summary->service_fee,
                'date' => $summary->created_at,
            ]);
            DB::commit();

            return redirect()->route('shop-collect-payment.index')->with('success', 'Shop payment recorded and delivery status updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Error processing payment: '.$e->getMessage());
        }
    }

    public function edit($id)
    {
        $summary = SummaryShopPayment::with(['confirmPayments', 'customerPayments', 'customer'])
            ->findOrFail($id);

        $existingPayments = $summary->confirmPayments->map(function ($p) {
            return [
                'amount' => $p->amount,
                'currency' => $p->currency,
                // 'payment_method' => $p->payment_method,
            ];
        });

        $deliveryIds = $summary->customerPayments->where('status', 'collected')
            ->pluck('delivery_id')->toArray();

        $deliveries = Delivery::with(['payments', 'customer'])->whereIn('id', $deliveryIds)->get();
        $customer = Customer::findOrFail($summary->customer_id);
        $branch = Branch::findOrFail(auth()->user()->default_branch_id);

        return view('pages.shop-collect-payment.view-collect-payment', [
            'summary' => $summary,
            'existingPayments' => $existingPayments,
            'exchangeRate' => $summary->exchange_rate,
            'invoiceNumber' => $summary->invoice_number,
            'customer' => $customer,
            'deliveries' => $deliveries,
            'branch' => $branch,
        ]);
    }

    public function update(Request $request, $id)
    {
        $request->validate([
            'customer_id' => 'required|exists:customers,id',
            'delivery_ids' => 'required|array',
            'delivery_ids.*' => 'exists:delivery,id',
            'payments' => 'required|array|min:1',
            'payments.*.amount' => 'required|numeric',
            'payments.*.currency' => 'required|in:usd,khr',
            'service_fee' => 'required|min:0',
            // 'payments.*.payment_method' => 'required|in:cash,transfer',
        ]);

        $summary = SummaryShopPayment::with('customer')->where('id', $id)->firstOrFail();

        try {
            DB::beginTransaction();
            $deliveryIds = $request->delivery_ids;
            $customerId = $request->customer_id;
            $serviceFee = $request->service_fee;
            $totalAmountUsd = $request->total_amount_usd;
            $totalAmountKhr = $request->total_amount_khr;

            $summary->update([
                'exchange_rate' => getExchangeRate(),
                'total_amount_usd' => $totalAmountUsd,
                'total_amount_khr' => $totalAmountKhr,
                'service_fee' => $serviceFee,
                'updated_by' => auth()->id(),
            ]);

            // Delete existing payments
            ConfirmShopPayment::where('summary_shop_payment_id', $summary->id)->delete();

            foreach ($request->payments as $payment) {
                ConfirmShopPayment::create([
                    'summary_shop_payment_id' => $summary->id,
                    'customer_id' => $customerId,
                    'amount' => $payment['amount'],
                    'currency' => $payment['currency'],
                    // 'payment_method' => $payment['payment_method'],
                ]);

            }

            // Delete & reinsert delivery links
            ShopCollectDeliveryPayment::where('summary_shop_payment_id', $summary->id)->delete();

            foreach ($deliveryIds as $deliveryId) {
                ShopCollectDeliveryPayment::create([
                    'summary_shop_payment_id' => $summary->id,
                    'customer_id' => $customerId,
                    'delivery_id' => $deliveryId,
                    // 'payment_date' => now(),
                    'note' => 'update payment with shop',
                    'created_by' => auth()->id(),
                ]);
            }

            DB::commit();

            return redirect()->route('shop-summary-payment.index')->with('success', 'Shop payment recorded and delivery status updated successfully.');
        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Error processing payment: '.$e->getMessage());
        }
    }

    public function invoice($invoice)
    {
        $summary = SummaryShopPayment::with(['confirmPayments', 'customerPayments', 'customer'])
            ->where('invoice_number', $invoice)
            ->firstOrFail();

        $existingPayments = $summary->confirmPayments->map(function ($p) {
            return [
                'amount' => $p->amount,
                'currency' => $p->currency,
                // 'payment_method' => $p->payment_method,
            ];
        });

        $deliveryIds = $summary->customerPayments->where('status', 'collected')
            ->pluck('delivery_id')->toArray();
        $deliveries = Delivery::with(['payments', 'customer'])->whereIn('id', $deliveryIds)->get();

        $deliveryStatus = $summary->customerPayments->whereIn('status', ['assigned', 'in_stock', 'canceled'])
            ->pluck('delivery_id')->toArray();
        $deliveryPending = Delivery::with(['payments', 'customer','cancelReason'])->whereIn('id', $deliveryStatus)->get();

        $customer = Customer::with('banks')->findOrFail($summary->customer_id);
        $branch = Branch::findOrFail($customer->branch_id ?? 1);

        return view('pages.shop-collect-payment.download', [
            'summary' => $summary,
            'existingPayments' => $existingPayments,
            'exchangeRate' => $summary->exchange_rate,
            'invoiceNumber' => $summary->invoice_number,
            'customer' => $customer,
            'deliveries' => $deliveries,
            'branch' => $branch,
            'deliveryPending' => $deliveryPending,
        ]);
    }

    public function download($invoice)
    {
        $summary = SummaryShopPayment::with(['confirmPayments', 'customerPayments', 'customer'])
            ->where('invoice_number', $invoice)
            ->firstOrFail();

        $existingPayments = $summary->confirmPayments->map(function ($p) {
            return [
                'amount' => $p->amount,
                'currency' => $p->currency,
            ];
        });

        $deliveryIds = $summary->customerPayments->where('status', 'collected')
            ->pluck('delivery_id')->toArray();
        $deliveries = Delivery::with(['payments', 'customer'])->whereIn('id', $deliveryIds)->get();

        $deliveryStatus = $summary->customerPayments->whereIn('status', ['assigned', 'in_stock', 'canceled'])
            ->pluck('delivery_id')->toArray();
        $deliveryPending = Delivery::with(['payments', 'customer','cancelReason'])->whereIn('id', $deliveryStatus)->get();

        $customer = Customer::with('banks')->findOrFail($summary->customer_id);

        $branch = Branch::findOrFail(auth()->user()->default_branch_id);

        $shopName = preg_replace('/[^A-Za-z0-9\-]/', '_', $customer->customer_name); // Replace special chars
        $fileName = 'Invoice_'.$shopName.'_'.$summary->invoice_number.'.pdf';

        $pdf = Pdf::setOption([
            'isRemoteEnabled' => true,
            'defaultFont' => 'Battambang',
        ])
        ->loadView('pages.shop-collect-payment.download', [
            'summary' => $summary,
            'existingPayments' => $existingPayments,
            'exchangeRate' => $summary->exchange_rate,
            'invoiceNumber' => $summary->invoice_number,
            'customer' => $customer,
            'deliveries' => $deliveries,
            'branch' => $branch,
            'deliveryPending' => $deliveryPending,
        ])
        ->setPaper('A4', 'landscape');


        return $pdf->download($fileName);
    }
}
