<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Enums\Status;
use App\Http\Requests\Delivery\ScanAssignRequest;
use App\Models\Delivery;
use App\Models\DeliveryScanAssign;
use App\Models\DeliveryStatusHistory;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class DeliveryScanAssignController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:delivery.assign')->only(['index', 'assignDriver']);
        $this->middleware('permission:delivery.assign-create')->only(['store']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = DeliveryScanAssign::with('delivery')
                ->where('status', 'pending')
                ->when($request->assign_to, function ($query, $assign_to) {
                    return $query->where('assign_to', $assign_to);
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('receiver_address', function ($row) {
                    return $row->delivery->receiver_address ?? 'N/A';
                })
                ->addColumn('receiver_phone', function ($row) {
                    return $row->delivery->receiver_phone ?? 'N/A';
                })
                ->addColumn('code', function ($row) {
                    return $row->delivery->code ?? 'N/A';
                })
                ->addColumn('amount_usd', function ($row) {
                    return $row->delivery->amount_usd;
                })
                ->addColumn('amount_khr', function ($row) {
                    return $row->delivery->amount_khr;
                })
                ->addColumn('created_by', function ($row) {
                    return $row->createdBy->display_name ?? 'N/A';
                })
                ->make(true);

        }

        return view('pages.delivery.scan-assign');
    }

    public function scanDriver()
    {
        $deliveryBy = User::get()->pluck('display_name', 'id');

        return view('pages.delivery.scan-assign', compact('deliveryBy'));
    }

    public function store(ScanAssignRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $deliveryById = $data['delivery_by'];
            $scan = DeliveryScanAssign::where('assign_to', $deliveryById)->where('status', Status::PENDING->value)->get();

            foreach ($scan as $delivery) {
                $delivery->update([
                    'status' => DeliveryStatus::ASSIGNED->value,
                ]);

                Delivery::where('id', $delivery->delivery_id)->update([
                    'delivery_by' => $deliveryById,
                    'status' => DeliveryStatus::ASSIGNED->value,
                    'assigned_at' => now(),
                    'updated_by' => auth()->id(),
                ]);

                DeliveryStatusHistory::create([
                    'delivery_id' => $delivery->delivery_id,
                    'status' => DeliveryStatus::ASSIGNED->value,
                    'changed_by' => auth()->id(),
                    'changed_at' => now(),
                    'note' => 'Sorting | Assigned to driver',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

            }

            DB::commit();

            return back()->with('success', 'Driver assigned successfully.');

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Failed to assign driver: '.$e->getMessage());

            return redirect()->back()->withInput()->with('error', 'Failed to assign driver. Please try again.');
        }
    }

    public function scan(Request $request)
    {
        $request->validate([
            'code' => 'required',
            'driver' => 'required',
        ]);

        // Find delivery first
        $delivery = Delivery::where('code', $request->code)
            ->whereNull('delivery_by')
            ->first();

        if (! $delivery) {
            return response()->json([
                'success' => false,
                'message' => 'Delivery not found or already assigned',
            ], 404);
        }

        // Check if already scanned
        $count = DeliveryScanAssign::where('delivery_id', $delivery->id)->count();
        if ($count > 0) {
            return response()->json([
                'success' => false,
                'message' => 'Already scanned',
            ], 409); // 409 Conflict is more correct
        }

        // Create scan assignment
        DeliveryScanAssign::create([
            'delivery_id' => $delivery->id,
            'assign_to' => $request->driver,
            'created_by' => auth()->user()->id,
            'status' => Status::PENDING->value,
        ]);

        return response()->json([
            'success' => true,
            'data' => $delivery,
        ]);
    }
}
