<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Http\Requests\Delivery\CancelReasonProcessRequest;
use App\Http\Requests\Delivery\PaymentProcessRequest;
use App\Http\Requests\Delivery\UpdatedRequest;
use App\Models\CancelReason;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryPayment;
use App\Models\DeliveryStatusHistory;
use App\Models\User;
use App\Models\ZonePrice;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use SimpleSoftwareIO\QrCode\Facades\QrCode;
use Yajra\DataTables\DataTables;

class DeliveryController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:delivery.view')->only(['index', 'show']);
        $this->middleware('permission:delivery.create')->only(['create', 'store']);
        $this->middleware('permission:delivery.edit')->only(['edit', 'update']);
        $this->middleware('permission:delivery.delete')->only(['destroy']);
        $this->middleware('permission:delivery.process-payment')->only(['processPayment']);
        $this->middleware('permission:delivery.process-cancel')->only(['processCancelReason']);
        $this->middleware('permission:delivery.print-invoice')->only(['printInvoice']);
    }

    public function deliveryAjax(Request $request)
    {
        $user = $request->user();
        $branchId = $user->default_branch_id;

        if ($request->ajax()) {
            $data = Delivery::where('branch_id', $branchId)
                ->where('is_active',true)
                ->when($user->hasRole('customer'), function ($query) use ($user) {
                    $query->where('customer_id', $user->customer_id);
                })
               
                ->when($user->hasRole('driver'), function ($query) use ($user) {
                    $query->where('delivery_by', $user->id);
                })

                // existing filters
                ->when($request->delivery_by, function ($query, $delivery_by) {
                    return $query->where('delivery_by', $delivery_by);
                })
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->status, function ($query, $status) {
                    return $query->where('status', $status);
                })
                ->when($request->entry_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->entry_at);
                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('created_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        return $query->whereBetween('created_at', [
                            $dates[0].' 00:00:00',
                            $dates[0].' 23:59:59',
                        ]);
                    }

                    return $query;
                })
                ->when($request->completed_at, function ($query) use ($request) {
                    $dates = explode(' to ', $request->completed_at);
                    if (count($dates) === 2) {
                        $from = $dates[0].' 00:00:00';
                        $to = $dates[1].' 23:59:59';

                        return $query->whereBetween('completed_at', [$from, $to]);
                    } elseif (count($dates) === 1 && $dates[0]) {
                        return $query->whereBetween('completed_at', [
                            $dates[0].' 00:00:00',
                            $dates[0].' 23:59:59',
                        ]);
                    }

                    return $query;
                })
                  ->orderByRaw("FIELD(status, ?, ?, ?, ?, ?, ?) ASC", [
                    DeliveryStatus::IN_STOCK->value,
                    DeliveryStatus::ASSIGNED->value,
                    DeliveryStatus::DELIVERED->value,
                    DeliveryStatus::COMPLETED->value,
                    DeliveryStatus::RETURNED->value,
                    DeliveryStatus::CANCELED->value,
                ])
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', fn ($row) => $row->customer->customer_name ?? 'N/A')
                ->addColumn('delivery_by', fn ($row) => $row->deliveryBy->display_name ?? 'N/A')
                ->addColumn('created_by', fn ($row) => $row->createdBy->display_name ?? 'N/A')
                ->editColumn('created_at', fn ($row) => $row->created_at ? $row->created_at->format('d/m/Y h:i A') : 'N/A')
                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);
                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                    HTML;
                })
                ->addColumn('action', function ($row) {
                    return view('pages.delivery.action-dropdown', [
                        'row' => $row,
                        'canEdit' => $row->is_paid !== true,
                    ])->render();
                })
                ->addColumn('status_cancel', function ($row) {
                    $hasReason = $row->cancel_reason_id !== null;
                    $reason = $hasReason ? 'Yes' : 'No';
                    $colorClass = $hasReason ? 'text-error' : 'text-success';
                    $tooltipText = $hasReason ? $row->cancelReason->reason : "Don't have cancelled";

                    // Escape quotes for Alpine
                    $tooltipTextEscaped = addslashes($tooltipText);

                    return <<<HTML
                        <div class="border badge space-x-2.5 {$colorClass}" x-tooltip.delay.500="'{$tooltipTextEscaped}'">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$reason}</span>
                        </div>
                    HTML;
                })

                ->rawColumns(['action', 'status_label', 'status_cancel'])
                ->make(true);
        }
    }

     public function deliveryAjaxDashboard(Request $request)
        {
            $user = $request->user();
            $branchId = $user->default_branch_id;

            if ($request->ajax()) {
                $data = Delivery::where('branch_id', $branchId)
                    ->where('status',DeliveryStatus::ASSIGNED->value)
                    ->when($user->hasRole('customer'), function ($query) use ($user) {
                        $query->where('customer_id', $user->customer_id);
                    })
                
                    ->when($user->hasRole('driver'), function ($query) use ($user) {
                        $query->where('delivery_by', $user->id);
                    })

                    // existing filters
                    ->when($request->delivery_by, function ($query, $delivery_by) {
                        return $query->where('delivery_by', $delivery_by);
                    })
                    ->when($request->customer_id, function ($query, $customer_id) {
                        return $query->where('customer_id', $customer_id);
                    })
                    ->when($request->status, function ($query, $status) {
                        return $query->where('status', $status);
                    })
                    ->when($request->entry_at, function ($query) use ($request) {
                        $dates = explode(' to ', $request->entry_at);
                        if (count($dates) === 2) {
                            $from = $dates[0].' 00:00:00';
                            $to = $dates[1].' 23:59:59';

                            return $query->whereBetween('created_at', [$from, $to]);
                        } elseif (count($dates) === 1 && $dates[0]) {
                            return $query->whereBetween('created_at', [
                                $dates[0].' 00:00:00',
                                $dates[0].' 23:59:59',
                            ]);
                        }

                        return $query;
                    })
                    ->when($request->completed_at, function ($query) use ($request) {
                        $dates = explode(' to ', $request->completed_at);
                        if (count($dates) === 2) {
                            $from = $dates[0].' 00:00:00';
                            $to = $dates[1].' 23:59:59';

                            return $query->whereBetween('completed_at', [$from, $to]);
                        } elseif (count($dates) === 1 && $dates[0]) {
                            return $query->whereBetween('completed_at', [
                                $dates[0].' 00:00:00',
                                $dates[0].' 23:59:59',
                            ]);
                        }

                        return $query;
                    })
                    ->orderBy('created_at', 'desc');

                return DataTables::of($data)
                    ->addColumn('customer_name', fn ($row) => $row->customer->customer_name ?? 'N/A')
                    ->addColumn('delivery_by', fn ($row) => $row->deliveryBy->display_name ?? 'N/A')
                    ->addColumn('created_by', fn ($row) => $row->createdBy->display_name ?? 'N/A')
                    ->editColumn('created_at', fn ($row) => $row->created_at ? $row->created_at->format('d/m/Y h:i A') : 'N/A')
                    ->addColumn('status_label', function ($row) {
                        $status = DeliveryStatus::tryFrom($row->status);
                        $label = $status?->label() ?? 'Unknown';

                        $colorClass = match ($status) {
                            DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                            DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                            DeliveryStatus::ASSIGNED => 'text-info',
                            DeliveryStatus::DELIVERED => 'text-success',
                            DeliveryStatus::COMPLETED => 'text-success',
                            DeliveryStatus::RETURNED => 'text-warning',
                            DeliveryStatus::CANCELED => 'text-error',
                            default => 'text-error',
                        };

                        return <<<HTML
                            <div class="badge space-x-2.5 {$colorClass}">
                                <div class="size-2 rounded-full bg-current"></div>
                                <span>{$label}</span>
                            </div>
                        HTML;
                    })
                    ->addColumn('action', function ($row) {
                        return view('pages.delivery.action-dropdown', [
                            'row' => $row,
                            'canEdit' => $row->is_paid !== true,
                        ])->render();
                    })
                    ->addColumn('status_cancel', function ($row) {
                        $hasReason = $row->cancel_reason_id !== null;
                        $reason = $hasReason ? 'Yes' : 'No';
                        $colorClass = $hasReason ? 'text-error' : 'text-success';
                        $tooltipText = $hasReason ? $row->cancelReason->reason : 'No cancel reason';

                        return <<<HTML
                            <div class="badge space-x-2.5 {$colorClass}" x-tooltip.delay.500="'{$tooltipText}'">
                                <div class="size-2 rounded-full bg-current"></div>
                                <span>{$reason}</span>
                            </div>
                        HTML;
                    })
                    ->rawColumns(['action', 'status_label', 'status_cancel'])
                    ->make(true);
            }
        }


    public function index(Request $request)
    {
        $branchId = $request->user()->default_branch_id;

        $deliveryBy = User::get()->pluck('display_name', 'id');
        $customer = Customer::active()
            ->where('branch_id', $branchId)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();

        $statusOptions = collect(DeliveryStatus::cases())->map(function ($status) {
            return [
                'value' => $status->value,
                'label' => $status->label(),
            ];
        });

        $cancelReasons = CancelReason::where('branch_id', $branchId)
            ->select('id', 'reason')
            ->get();

        return view('pages.delivery.index', compact('deliveryBy', 'customer', 'statusOptions', 'cancelReasons'));
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $id)
    {
        $delivery = Delivery::with(['customer', 'statuses', 'payments'])->findOrFail($id);
        $tracking = route('tracking.search');
        $qrCodeBase64 = base64_encode(
            QrCode::format('png')->size(200)->generate($tracking.'?code='.$delivery->code)
        );

        return view('pages.delivery.view', compact('delivery', 'qrCodeBase64'));
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(string $id)
    {
        $branch = auth()->user()->default_branch_id;
        $pickupBy = User::get()->pluck('display_name', 'id');
        $customer = Customer::active()
            ->where('branch_id', $branch)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();

        $zone = ZonePrice::where('status', 'active')
            ->where('branch_id', $branch)
            ->select('id', 'zone_name', 'price', 'currency')
            ->get();
        $delivery = Delivery::with(['customer', 'statuses'])->findOrFail($id);

        return view('pages.delivery.inquiry.edit', compact('delivery', 'pickupBy', 'customer', 'zone'));
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(UpdatedRequest $request, string $id)
    {
        $data = $request->validated();
        DB::beginTransaction();
        try {
            $delivery = Delivery::findOrFail($id);
            $delivery->zone_price_id = $data['zone_price_id'] ?? null;
            $delivery->receiver_address = $data['receiver_address'];
            $delivery->delivery_price = $data['delivery_price'];
            $delivery->receiver_phone = $data['receiver_phone'];
            $delivery->amount_usd = $data['amount_usd'];
            $delivery->amount_khr = $data['amount_khr'];
            $delivery->type_of_delivery = $data['type_of_delivery'];
            $delivery->type_of_cod = $data['type_of_cod'] ?? 'No';
            $delivery->note = $data['note'] ?? null;
            $delivery->updated_by = auth()->user()->id;
            $delivery->any_extra_fee = $data['any_extra_fee'];
            $delivery->who_pay_fee = $data['who_pay_fee'];
            $delivery->save();
            DB::commit();

            return redirect()->back()->with('success', 'delivery updated successfully.');

        } catch (\Exception $e) {
            DB::rollBack();

            return redirect()->back()->with('error', 'Failed to update delivery. '.$e->getMessage());
        }
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(string $id)
    {
        //
    }

    public function processPayment(PaymentProcessRequest $request)
    {
        $data = $request->validated();
       
        DB::beginTransaction();

        try {
            $delivery = Delivery::findOrFail($data['delivery_id']);
            $delivery->status = DeliveryStatus::COMPLETED->value;
            $delivery->completed_at = now();
            $delivery->note = $data['note'] ?? null;
            $delivery->save();

            $customer = $delivery->customer?->customer_name ?? 'N/A';

            DeliveryStatusHistory::create([
                'delivery_id' => $data['delivery_id'],
                'status' => DeliveryStatus::COMPLETED->value,
                'note' => $data['note'] ?? 'delivery completed by '.auth()->user()->display_name,
                'changed_by' => auth()->user()->id,
                'changed_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);

            foreach ($data['payments'] as $paymentTo) {
                $payment = $paymentTo['payment_to'];
                $amount = $paymentTo['amount'];
                $currency = $paymentTo['currency'];
                $paymentMethod = $paymentTo['payment_method'];

                DeliveryPayment::create([
                    'delivery_id' => $data['delivery_id'],
                    'payment_to' => $payment,
                    'amount' => $amount,
                    'currency' => $currency,
                    'payment_method' => $paymentMethod,
                    'note' => $data['note'] ?? null,
                    'created_by' => auth()->user()->id,
                ]);

                sendTelegramNotification([
                    'delivery_code' => $delivery->code,
                    'customer' => $customer,
                    'payment' => $payment,
                    'amount' => $amount,
                    'currency' => $currency,
                    'payment_method' => $paymentMethod,
                ]);

            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Payment processed successfully.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Payment error: '.$e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong.',
            ], 500);
        }
    }

    public function printInvoice(string $id)
    {
        $delivery = Delivery::with(['customer', 'statuses', 'payments'])->findOrFail($id);
        $tracking = route('tracking.search');
        $qrCodeBase64 = base64_encode(
            QrCode::format('png')->size(200)->generate($tracking.'?code='.$delivery->code)
        );

        return view('pages.delivery.print-invoice', compact('delivery', 'qrCodeBase64'));
    }

    public function processCancelReason(CancelReasonProcessRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $reason = CancelReason::findOrFail($data['reason']);
            $message = $data['note'] ?? $reason->reason;

            $delivery = Delivery::findOrFail($data['delivery_id']);
            $delivery->status = DeliveryStatus::CANCELED->value;
            $delivery->cancel_reason_id = $data['reason'];
            $delivery->cancel_note =  $reason;
            $delivery->canceled_at = now();
            $delivery->save();

           
            DeliveryStatusHistory::create([
                'delivery_id' => $data['delivery_id'],
                'status' => DeliveryStatus::CANCELED->value,
                'note' => $message,
                'changed_by' => auth()->user()->id,
                'changed_at' => now(),
                'created_at' => now(),
                'updated_at' => now(),
            ]);
            $customer = $delivery->customer?->customer_name ?? 'N/A';
            sendTelegramCancelNotification([
                'delivery_code' => $delivery->code,
                'customer' => $customer,
                'reason' => $message,
            ]);

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Cancel Reason processed successfully.',
            ]);
        } catch (\Exception $e) {
            DB::rollBack();
            \Log::error('Cancel Reason error: '.$e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Something went wrong while processing the Cancel Reason. Please try again later.',
            ], 500);
        }
    }

    public function getExpressFee(Request $request)
    {
        $zoneId = $request->query('zone_id');
        $fee = ZonePrice::with('zoneType')->where('id', $zoneId)
            ->first();
        $expressFee = $fee->zoneType?->express ?? 0;

        return response()->json([
            'express_fee' => $expressFee,
        ]);
    }
}
