<?php

namespace App\Http\Controllers;

use App\Enums\DeliveryStatus;
use App\Enums\Status;
use App\Models\CancelReason;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryPayment;
use Carbon\Carbon;

class DashboardController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:dashboard.view')->only(['index']);
    }

    public function index()
    {

        $branchId = auth()->user()->default_branch_id;
        $monthlyFees = Delivery::selectRaw('
            MONTH(completed_at) as month,
            SUM(delivery_price) as total_fee
        ')
            ->where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_paid', true)
            ->whereNotNull('completed_at')
            ->groupByRaw('MONTH(completed_at)')
            ->orderByRaw('MONTH(completed_at)')
            ->get();

        $profit = array_fill(0, 12, 0);

        foreach ($monthlyFees as $item) {
            $profit[$item->month - 1] = (float) $item->total_fee;
        }

        $currentMonthTotal = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_paid', true)
            ->whereNotNull('completed_at')
            ->whereMonth('completed_at', now()->month)
            ->whereYear('completed_at', now()->year)
            ->sum('delivery_price');

        $lastMonth = now()->subMonth();
        $lastMonthTotal = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_paid', true)
            ->whereNotNull('completed_at')
            ->whereMonth('completed_at', $lastMonth->month)
            ->whereYear('completed_at', $lastMonth->year)
            ->sum('delivery_price');

        $percentageChange = null;
        $trend = null;

        if ($lastMonthTotal > 0) {
            $change = $currentMonthTotal - $lastMonthTotal;
            $percentageChange = round(($change / $lastMonthTotal) * 100, 1);
            $trend = $percentageChange > 0 ? 'up' : ($percentageChange < 0 ? 'down' : 'same');
        }

        $exchangeRate = getExchangeRate();

        $totalUsdThisMonth = Delivery::selectRaw('
                SUM(amount_usd + (amount_khr / ?)) as total_usd
            ', [$exchangeRate])
            ->where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->where('is_paid', true)
            ->whereNotNull('completed_at')
            ->whereMonth('completed_at', now()->month)
            ->whereYear('completed_at', now()->year)
            ->value('total_usd');

        $countCompleted = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::COMPLETED->value)
            ->whereMonth('completed_at', now()->month)
            ->whereYear('completed_at', now()->year)
            ->count();

        $countInStock = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::IN_STOCK->value)
            ->whereMonth('created_at', now()->month)
            ->whereYear('created_at', now()->year)
            ->count();

        $countDelivery = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::ASSIGNED->value)
            ->whereMonth('assigned_at', now()->month)
            ->whereYear('assigned_at', now()->year)
            ->count();
        $countCanceled = Delivery::where('branch_id', $branchId)
            ->where('status', DeliveryStatus::CANCELED->value)
            ->whereMonth('canceled_at', now()->month)
            ->whereYear('canceled_at', now()->year)
            ->count();

        $countCustomer = Customer::where('branch_id', $branchId)
            ->where('status', Status::ACTIVE->value)
            ->count();

         $cancelReasons = CancelReason::where('branch_id', $branchId)
            ->select('id', 'reason')
            ->get();

        /**
         * For Driver Role
         */
        $driver = auth()->user();
        // Base delivery query
        $deliveryQuery = Delivery::where('delivery_by', $driver->id)
        ->where('is_active',true);

        /* ============================
         * Payments (FROM driver_payments)
         * ============================ */
            $totalOrginCollectedUSD  = (clone $deliveryQuery)
                ->where('is_paid',false)
                ->whereIn('status',[DeliveryStatus::ASSIGNED->value,DeliveryStatus::COMPLETED->value])
                ->sum('amount_usd');
         
            $totalOrginCollectedKHR  = (clone $deliveryQuery)
                ->where('is_paid',false)
                ->whereIn('status',[DeliveryStatus::ASSIGNED->value,DeliveryStatus::COMPLETED->value])
                ->sum('amount_khr');

            /* ============================
            * Currency-wise Collected Summary
            * ============================ */
            $totalCollect = DeliveryPayment::whereHas('delivery', function ($q) use ($driver) {
                $q->where('delivery_by', $driver->id)->where('is_paid',false);
            })
            ->selectRaw('currency, SUM(amount) as total_amount')
            ->where('payment_to','driver')
            ->groupBy('currency')
            ->get();


            $collectedUSD = 0;
            $collectedKHR = 0;

            foreach ($totalCollect as $row) {
                if (strtolower($row->currency) === 'usd') {
                    $collectedUSD = (float) $row->total_amount;
                }

                if (strtolower($row->currency) === 'khr') {
                    $collectedKHR = (float) $row->total_amount;
                }
            }
            $exchangeRate = getExchangeRate();

            // Original (unpaid COD)
            $originalTotalInUSD =
                (float) $totalOrginCollectedUSD +
                ((float) $totalOrginCollectedKHR / $exchangeRate);

            // Collected
            $collectedTotalInUSD =
                $collectedUSD +
                ($collectedKHR / $exchangeRate);

            // Outstanding
            $outstanding = max(
                round($originalTotalInUSD - $collectedTotalInUSD, 2),
                0
            );


        /* ============================
         * Today Summary
         * ============================ */
        $today = Carbon::today();

        $todaySummary = [
            'total_package' => (clone $deliveryQuery)
             ->where('is_paid', false)
               ->where(function ($q) use ($today) {
                    $q->whereDate('created_at', $today)
                    ->orWhereDate('assigned_at', $today)
                    ->orWhereDate('completed_at', $today)
                    ->orWhereDate('canceled_at', $today)
                    ->where('status', DeliveryStatus::ASSIGNED->value);
                })

                ->count(),

            'in_progress' =>  (clone $deliveryQuery)
                ->where('status', DeliveryStatus::ASSIGNED->value)
                ->orWhereDate('assigned_at', $today)
                ->count(),


            'completed' => (clone $deliveryQuery)
                ->where('status', 'completed')
                ->whereDate('completed_at', $today)
                ->where('is_paid', false)
                ->count(),

            'cancelled' => (clone $deliveryQuery)
                ->where('status', 'canceled')
                ->whereDate('canceled_at', $today)
                ->count(),
        ];

       $driverSummary = ([
            'payment' => [
                'total_original' => [
                    'usd' => (float) $totalOrginCollectedUSD,
                    'khr' => (float) $totalOrginCollectedKHR,
                ],
                'total_collected' => [
                    'usd' => $collectedUSD,
                    'khr' => $collectedKHR,
                ],
                'outstanding_usd' => $outstanding,
            ],
            'today' => $todaySummary,
        ]);



        return view('pages.dashboard.index', compact('profit',
            'currentMonthTotal', 'lastMonthTotal',
            'percentageChange', 'trend', 'totalUsdThisMonth',
            'countCompleted',
            'countInStock',
            'countDelivery',
            'countCanceled',
            'countCustomer',
            'cancelReasons',
            'driverSummary'
            ));
    }
}
