<?php

namespace App\Http\Controllers;

use App\Enums\BookingStatus;
use App\Enums\DeliveryStatus;
use App\Http\Requests\Delivery\AssignDriverRequest;
use App\Models\Customer;
use App\Models\Delivery;
use App\Models\DeliveryStatusHistory;
use App\Models\User;
use App\Models\ZonePrice;
use App\Models\ZoneType;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Yajra\DataTables\DataTables;

class AssignDriverController extends Controller
{
    /**
     * @method void middleware($middleware, array|string $options = [])
     */
    public function __construct()
    {
        $this->middleware('permission:delivery.assign')->only(['index', 'assignDriver']);
        $this->middleware('permission:delivery.assign-create')->only(['store']);
    }

    /**
     * Display a listing of the resource.
     */
    public function index(Request $request)
    {
        if ($request->ajax()) {
            $data = Delivery::where('branch_id', auth()->user()->default_branch_id)
                ->whereIn('status', [
                    DeliveryStatus::IN_STOCK->value,
                    BookingStatus::ARRIVEDWAREHOUSE->value,
                    BookingStatus::ARRIVEDWAREHOUSEPP->value,
                ])
                ->when($request->customer_id, function ($query, $customer_id) {
                    return $query->where('customer_id', $customer_id);
                })
                ->when($request->zone_type_id, function ($query, $zone_type_id) {
                    return $query->whereHas('zonePrice', function ($q) use ($zone_type_id) {
                        $q->where('zone_type_id', $zone_type_id);
                    });
                })
                ->orderBy('created_at', 'desc');

            return DataTables::of($data)
                ->addColumn('customer_name', function ($row) {
                    return $row->customer->customer_name ?? 'N/A';
                })
                ->addColumn('pickup_by', function ($row) {
                    return $row->pickupBy->display_name ?? 'N/A';
                })

                ->addColumn('created_by', function ($row) {
                    return $row->createdBy->display_name ?? 'N/A';
                })
                ->editColumn('pickup_at', function ($row) {
                    if (! $row->pickup_at) {
                        return 'N/A';
                    }

                    return \Carbon\Carbon::parse($row->pickup_at)->format('d/m/Y h:i A');
                })
                ->addColumn('status_label', function ($row) {
                    $status = DeliveryStatus::tryFrom($row->status);

                    $label = $status?->label() ?? 'Unknown';

                    $colorClass = match ($status) {
                        DeliveryStatus::IN_STOCK => 'text-slate-800 dark:text-navy-100',
                        DeliveryStatus::BOOKING => 'text-primary dark:text-accent-light',
                        DeliveryStatus::ASSIGNED => 'text-info',
                        DeliveryStatus::DELIVERED => 'text-success',
                        DeliveryStatus::COMPLETED => 'text-success',
                        DeliveryStatus::RETURNED => 'text-warning',
                        DeliveryStatus::CANCELED => 'text-error',
                        default => 'text-error',
                    };

                    return <<<HTML
                        <div class="badge space-x-2.5 {$colorClass}">
                            <div class="size-2 rounded-full bg-current"></div>
                            <span>{$label}</span>
                        </div>
                        HTML;
                })
                ->rawColumns(['status_label'])
                ->make(true);

        }

        return view('pages.delivery.assign-driver');
    }

    public function assignDriver()
    {
        $deliveryBy = User::get()->pluck('display_name', 'id');

        $customer = Customer::active()
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'customer_name', 'phone', 'currency')
            ->get();
        $zoneTypes = ZoneType::where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'zone_type')
            ->get();
        $zone = ZonePrice::where('status', 'active')
            ->where('branch_id', auth()->user()->default_branch_id)
            ->select('id', 'zone_name', 'price', 'currency')
            ->get();

        return view('pages.delivery.assign-driver', compact('deliveryBy', 'customer', 'zone', 'zoneTypes'));
    }

    public function store(AssignDriverRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $deliveryIds = $data['delivery_ids'];
            $deliveryById = $data['delivery_by'];

            $deliveries = Delivery::whereIn('id', $deliveryIds)
                ->whereIn('status', [
                    DeliveryStatus::IN_STOCK->value,
                    BookingStatus::ARRIVEDWAREHOUSE->value,
                    BookingStatus::ARRIVEDWAREHOUSEPP->value,
                ])
                ->get();

            foreach ($deliveries as $delivery) {
                $delivery->update([
                    'delivery_by' => $deliveryById,
                    'status' => DeliveryStatus::ASSIGNED->value,
                    'assigned_at' => now(),
                    'updated_by' => auth()->id(),
                ]);

                DeliveryStatusHistory::create([
                    'delivery_id' => $delivery->id,
                    'status' => DeliveryStatus::ASSIGNED->value,
                    'changed_by' => auth()->id(),
                    'changed_at' => now(),
                    'note' => 'Sorting | Assigned to driver',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

            }

            DB::commit();

            return back()->with('success', 'Driver assigned successfully.');

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Failed to assign driver: '.$e->getMessage());

            return redirect()->back()->withInput()->with('error', 'Failed to assign driver. Please try again.');
        }
    }
}
