<?php

namespace App\Http\Controllers\Api;

use App\Enums\DeliveryStatus;
use App\Enums\Status;
use App\Http\Controllers\Controller;
use App\Http\Requests\Api\Delivery\AssignPackageRequest;
use App\Http\Requests\Api\Delivery\DriverListingRequest;
use App\Http\Requests\Api\Delivery\ScanRequest;
use App\Models\Delivery;
use App\Models\DeliveryScanAssign;
use App\Models\DeliveryStatusHistory;
use App\Models\User;
use App\Traits\ApiResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;

class DriverController extends Controller
{
    use ApiResponse;

    /**
     * Driver Listing
     *
     * @return \Illuminate\Http\JsonResponse
     */
    public function driverListing(Request $request)
    {
        $branchId = $request->user()->default_branch_id;

        $drivers = User::role('driver')
            ->where('default_branch_id', $branchId)
            ->where('status', Status::ACTIVE->value)
            ->select('id', 'display_name', 'name', 'phone')
            ->paginate(30);

        return $this->successResponse($drivers, 'driver listing');
    }

    /**
     * Scan get product.
     */
    public function scan(ScanRequest $request)
    {

        $validated = $request->validated();
        $delivery = Delivery::where('code', $validated['code'])
            ->whereNull('delivery_by')
            ->first();

        if (! $delivery) {
            return $this->errorResponse('Delivery not found or already assigned',
                'Delivery not found or already assigned', 422);
        }

        // Check if already scanned
        $count = DeliveryScanAssign::where('delivery_id', $delivery->id)->count();
        if ($count > 0) {
            return $this->errorResponse('Already scaned', 'Already scaned', 422);
        }

        $scan = DeliveryScanAssign::create([
            'delivery_id' => $delivery->id,
            'assign_to' => $validated['driver'],
            'created_by' => auth()->user()->id,
            'status' => Status::PENDING->value,
        ]);

        return $this->successResponse($scan, 'scan success');
    }

    /**
     * Scan listing all.
     */
    public function scanListing(DriverListingRequest $request)
    {

        $validated = $request->validated();
        $driverId = $validated['driver_id'];

        $data = DeliveryScanAssign::with('delivery')
            ->where('status', Status::PENDING->value)
            ->where('assign_to', $driverId)
            ->orderBy('created_at', 'desc')
            ->paginate(30);

        $data->getCollection()->transform(function ($item) {
            return [
                'delivery_id' => $item->delivery_id,
                'status' => $item->status,
                'created_at' => $item->created_at,
                // delivery fields
                'code' => $item->delivery->code ?? null,
                'receiver_address' => $item->delivery->receiver_address ?? null,
                'receiver_phone' => $item->delivery->receiver_phone ?? null,
                'amount_usd' => $item->delivery->amount_usd ?? 0,
                'amount_khr' => $item->delivery->amount_khr ?? 0,
            ];
        });

        return $this->successResponse($data, 'driver listing');
    }

    /**
     * Assign package to driver
     */
    public function assignPackage(AssignPackageRequest $request)
    {
        $data = $request->validated();

        DB::beginTransaction();

        try {
            $deliveryById = $data['driver'];
            $scan = DeliveryScanAssign::where('assign_to', $deliveryById)->where('status', Status::PENDING->value)->get();

            foreach ($scan as $delivery) {
                $delivery->update([
                    'status' => DeliveryStatus::ASSIGNED->value,
                ]);

                Delivery::where('id', $delivery->delivery_id)->update([
                    'delivery_by' => $deliveryById,
                    'status' => DeliveryStatus::ASSIGNED->value,
                    'assigned_at' => now(),
                    'updated_by' => auth()->id(),
                ]);

                DeliveryStatusHistory::create([
                    'delivery_id' => $delivery->delivery_id,
                    'status' => DeliveryStatus::ASSIGNED->value,
                    'changed_by' => auth()->id(),
                    'changed_at' => now(),
                    'note' => 'Sorting | Assigned to driver',
                    'created_at' => now(),
                    'updated_at' => now(),
                ]);

            }

            DB::commit();

            return $this->successResponse([], 'Assigned success');

        } catch (\Exception $e) {
            DB::rollBack();

            return $this->errorResponse('Failed to assign delivery '.$e->getMessage(), 500);
        }
    }
}
